const base_url = 'https://wxxcx.iyouwc.cn:2026';
const static_base = 'https://image.iyouwc.cn';

/**
 * 发起 HTTP 请求
 */
export async function http_request(api, data = {}, need_token = true) {
	return new Promise((resolve, reject) => {
		console.log(`[HTTP] 📤 请求接口: ${api}`, data);
		const header = {
			'Content-Type': 'application/json'
		};
		if (need_token) {
			const token = uni.getStorageSync('access_token');
			if (token) {
				header.Authorization = `Bearer ${token}`;
			} else {
				console.log('[HTTP] ❌ 未登录');
				reject({
					code: 401,
					message: '未登录'
				});
				return;
			}
		}
		const request = () => {
			uni.request({
				url: `${base_url}/${api}`,
				method: 'POST',
				header,
				data,
				success: async (res) => {
					if (res.statusCode === 200) {
						switch (res.data.code) {
							case 200:
								console.log(`[HTTP] ✅ 请求成功: ${api}`, res.data);
								resolve(res.data);
								break;

							case 500:
								console.log(`[HTTP] ❌ 服务器错误: ${api}`, res.data);
								uni.showToast({
									icon: 'none',
									title: res.data.message || '服务器错误'
								});
								resolve(res.data);
								break;

							case 401:
								console.log(res.data)
								console.log(`[HTTP] 🔑 Token过期，准备刷新...`);
								try {
									await refresh_token_def();
									console.log(`[HTTP] 🔄 刷新成功，重试请求: ${api}`);
									await new Promise(resolve => setTimeout(resolve, 500))
									return await http_request(api, data, true);
								} catch (err) {
									console.log('[HTTP] ❌ 刷新失败，跳转登录');
									uni.removeStorageSync('access_token');
									uni.removeStorageSync('refresh_token');
									uni.removeStorageSync('token_expire');
									// 跳转登录
									uni.redirectTo({
										url: '/pages/profile/login/login?showToastTitle=需要重新登录'
									});
									throw {
										code: 401,
										message: '认证失败，已跳转登录'
									};
								}
								break;

							case 403:
								console.log(`[HTTP] 🔒 身份无效，跳转登录`);
								uni.redirectTo({
									url: '/pages/profile/login/login?showToastTitle=需要重新登录'
								});
								resolve(res.data);
								break;

							default:
								console.log(`[HTTP] 🔔 未知业务码: ${api}, code=${res.data.code}`);
								resolve(res.data);
								break;
						}
					} else {
						console.log(`[HTTP] 🌐 HTTP状态错误: ${res.statusCode}`);
						uni.showToast({
							title: `请求异常: ${res.statusCode}`,
							icon: 'none'
						});
						reject(res);
					}
				},
				fail: (err) => {
					console.error(`[HTTP] ❌ 请求失败 [${api}]`, err);
					uni.showToast({
						title: '网络异常',
						icon: 'none'
					});
					reject({
						code: 500,
						message: '网络请求失败',
						error: err
					});
				}
			});
		};

		// 执行请求
		request();
	});
}

/**
 * 刷新 Token
 */
export async function refresh_token_def() {
	return new Promise((resolve, reject) => {
		const refresh_token = uni.getStorageSync('refresh_token');
		if (!refresh_token) {
			console.log('[REFRESH] ❌ 无 refresh_token 跳转登录');
			uni.reLaunch({
				url: '/pages/profile/login/login?showToastTitle=请先登录'
			});
			reject();
		}

		console.log('[REFRESH] 🔄 正在刷新 access_token...');
		uni.request({
			url: `${base_url}/get_token`,
			method: 'POST',
			header: {
				'Content-Type': 'application/json',
				Authorization: `Bearer ${refresh_token}`
			},
			success: async (res) => {
				if (res.data.code === 200) {
					const {
						access_token,
						refresh_token: new_refresh_token,
						expires_in
					} = res.data.data;
					uni.setStorageSync('access_token', access_token);
					uni.setStorageSync('refresh_token', new_refresh_token);
					uni.setStorageSync('token_expire', Date.now() + expires_in * 1000);
					console.log('[REFRESH] ✅ Token 刷新成功');
					resolve();
				} else {
					console.log('[REFRESH] ❌ 刷新失败 跳转登录', res);
					uni.reLaunch({
						url: '/pages/profile/login/login?showToastTitle=请先登录'
					});
					reject();
				}
			},
			fail: (err) => {
				console.error('[REFRESH] ❌ 请求失败 跳转登录', err);
				uni.reLaunch({
					url: '/pages/profile/login/login?showToastTitle=请先登录'
				});
				reject(err);
			}
		});
	});
}


export function upload_file(temp_file_path, options = {}) {
	return new Promise((resolve, reject) => {
		// 读取 token
		const access_token = uni.getStorageSync('access_token');
		if (!access_token) {
			console.log('[UPLOAD] ❌ 未登录');
			uni.showToast({
				title: '请先登录',
				icon: 'none'
			});
			reject({
				code: 401,
				message: '未登录'
			});
			return;
		}

		const {
			path = 'tmp',
				name = 'file',
				on_progress = null
		} = options;


		console.log(`[UPLOAD] 📤 开始上传文件: ${temp_file_path}`),
			console.log(`请求地址: ${base_url}/upload_image`)
		console.log(`token ${access_token}`)
		console.log(`formData: {'upload_path':${path}};`)

		const upload_task = uni.uploadFile({
			url: `${base_url}/upload_image`,
			filePath: temp_file_path,
			name: 'file',
			header: {
				'Authorization': `Bearer ${access_token}`
			},
			formData: {
				upload_path: path
			},
			success: async (res) => {
				console.log('✅ 上传成功 - 原始响应:', res);
				let response_data;
				try {
					response_data = JSON.parse(res.data);
					console.log('🧠 解析后的响应:', response_data);
				} catch (e) {
					console.error('❌ 解析失败:', e);
					uni.showToast({
						title: '解析失败',
						icon: 'none'
					});
					reject({
						success: false,
						message: '服务器返回格式错误'
					});
					return;
				}

				// 判断返回状态
				if (response_data.code === 200) {
					// ✅ 上传成功，返回完整图片 URL
					const image_url = `${static_base}/${response_data.url}`;
					resolve({
						success: true,
						data: {
							...response_data,
							image_url, // 完整可访问 URL
							relative_url: response_data.url // 相对路径
						}
					});
				} else if (response_data.code === 401) {
					console.log(`[HTTP] 🔑 Token过期，准备刷新...`);
					try {
						await refresh_token_def();
						console.log(`[HTTP] 🔄 刷新成功，重试请求: ${api}`);
						return await upload_file(temp_file_path, options)
					} catch (err) {
						console.log('[HTTP] ❌ 刷新失败，跳转登录');
						uni.removeStorageSync('access_token');
						uni.removeStorageSync('refresh_token');
						uni.removeStorageSync('token_expire');
						// 跳转登录
						uni.redirectTo({
							url: '/pages/profile/login/login?showToastTitle=需要重新登录'
						});
						throw {
							code: 401,
							message: '认证失败，已跳转登录'
						};
					}
				} else {
					uni.showToast({
						title: response_data.message || '上传失败',
						icon: 'none',
						duration: 2000
					});
					reject({
						success: false,
						message: response_data.message
					});
				}
			},
			fail: (err) => {
				console.error('上传失败:', err);
				uni.showToast({
					title: '网络错误或服务器无响应',
					icon: 'none'
				});
				reject({
					success: false,
					message: '网络错误'
				});
			}
		});
		// 监听上传进度
		if (typeof on_progress === 'function') {
			upload_task.onProgressUpdate((progress_res) => {
				on_progress(
					progress_res.progress, // 当前进度 %
					progress_res.totalBytesSent, // 已发送字节数
					progress_res.totalBytesExpectedToSend // 总字节数
				);
			});
		}
	});
}