"use strict";
const common_vendor = require("../common/vendor.js");
const base_url = "https://wxxcx.iyouwc.cn:2026";
const static_base = "https://image.iyouwc.cn";
async function http_request(api2, data = {}, need_token = true) {
  return new Promise((resolve, reject) => {
    common_vendor.index.__f__("log", "at utils/request.js:9", `[HTTP] 📤 请求接口: ${api2}`, data);
    const header = {
      "Content-Type": "application/json"
    };
    if (need_token) {
      const token = common_vendor.index.getStorageSync("access_token");
      if (token) {
        header.Authorization = `Bearer ${token}`;
      } else {
        common_vendor.index.__f__("log", "at utils/request.js:18", "[HTTP] ❌ 未登录");
        reject({
          code: 401,
          message: "未登录"
        });
        return;
      }
    }
    const request = () => {
      common_vendor.index.request({
        url: `${base_url}/${api2}`,
        method: "POST",
        header,
        data,
        success: async (res) => {
          if (res.statusCode === 200) {
            switch (res.data.code) {
              case 200:
                common_vendor.index.__f__("log", "at utils/request.js:36", `[HTTP] ✅ 请求成功: ${api2}`, res.data);
                resolve(res.data);
                break;
              case 500:
                common_vendor.index.__f__("log", "at utils/request.js:41", `[HTTP] ❌ 服务器错误: ${api2}`, res.data);
                common_vendor.index.showToast({
                  icon: "none",
                  title: res.data.message || "服务器错误"
                });
                resolve(res.data);
                break;
              case 401:
                common_vendor.index.__f__("log", "at utils/request.js:50", res.data);
                common_vendor.index.__f__("log", "at utils/request.js:51", `[HTTP] 🔑 Token过期，准备刷新...`);
                try {
                  await refresh_token_def();
                  common_vendor.index.__f__("log", "at utils/request.js:54", `[HTTP] 🔄 刷新成功，重试请求: ${api2}`);
                  await new Promise((resolve2) => setTimeout(resolve2, 500));
                  return await http_request(api2, data, true);
                } catch (err) {
                  common_vendor.index.__f__("log", "at utils/request.js:58", "[HTTP] ❌ 刷新失败，跳转登录");
                  common_vendor.index.removeStorageSync("access_token");
                  common_vendor.index.removeStorageSync("refresh_token");
                  common_vendor.index.removeStorageSync("token_expire");
                  common_vendor.index.redirectTo({
                    url: "/pages/profile/login/login?showToastTitle=需要重新登录"
                  });
                  throw {
                    code: 401,
                    message: "认证失败，已跳转登录"
                  };
                }
                break;
              case 403:
                common_vendor.index.__f__("log", "at utils/request.js:74", `[HTTP] 🔒 身份无效，跳转登录`);
                common_vendor.index.redirectTo({
                  url: "/pages/profile/login/login?showToastTitle=需要重新登录"
                });
                resolve(res.data);
                break;
              default:
                common_vendor.index.__f__("log", "at utils/request.js:82", `[HTTP] 🔔 未知业务码: ${api2}, code=${res.data.code}`);
                resolve(res.data);
                break;
            }
          } else {
            common_vendor.index.__f__("log", "at utils/request.js:87", `[HTTP] 🌐 HTTP状态错误: ${res.statusCode}`);
            common_vendor.index.showToast({
              title: `请求异常: ${res.statusCode}`,
              icon: "none"
            });
            reject(res);
          }
        },
        fail: (err) => {
          common_vendor.index.__f__("error", "at utils/request.js:96", `[HTTP] ❌ 请求失败 [${api2}]`, err);
          common_vendor.index.showToast({
            title: "网络异常",
            icon: "none"
          });
          reject({
            code: 500,
            message: "网络请求失败",
            error: err
          });
        }
      });
    };
    request();
  });
}
async function refresh_token_def() {
  return new Promise((resolve, reject) => {
    const refresh_token = common_vendor.index.getStorageSync("refresh_token");
    if (!refresh_token) {
      common_vendor.index.__f__("log", "at utils/request.js:122", "[REFRESH] ❌ 无 refresh_token 跳转登录");
      common_vendor.index.reLaunch({
        url: "/pages/profile/login/login?showToastTitle=请先登录"
      });
      reject();
    }
    common_vendor.index.__f__("log", "at utils/request.js:129", "[REFRESH] 🔄 正在刷新 access_token...");
    common_vendor.index.request({
      url: `${base_url}/get_token`,
      method: "POST",
      header: {
        "Content-Type": "application/json",
        Authorization: `Bearer ${refresh_token}`
      },
      success: async (res) => {
        if (res.data.code === 200) {
          const {
            access_token,
            refresh_token: new_refresh_token,
            expires_in
          } = res.data.data;
          common_vendor.index.setStorageSync("access_token", access_token);
          common_vendor.index.setStorageSync("refresh_token", new_refresh_token);
          common_vendor.index.setStorageSync("token_expire", Date.now() + expires_in * 1e3);
          common_vendor.index.__f__("log", "at utils/request.js:147", "[REFRESH] ✅ Token 刷新成功");
          resolve();
        } else {
          common_vendor.index.__f__("log", "at utils/request.js:150", "[REFRESH] ❌ 刷新失败 跳转登录", res);
          common_vendor.index.reLaunch({
            url: "/pages/profile/login/login?showToastTitle=请先登录"
          });
          reject();
        }
      },
      fail: (err) => {
        common_vendor.index.__f__("error", "at utils/request.js:158", "[REFRESH] ❌ 请求失败 跳转登录", err);
        common_vendor.index.reLaunch({
          url: "/pages/profile/login/login?showToastTitle=请先登录"
        });
        reject(err);
      }
    });
  });
}
function upload_file(temp_file_path, options = {}) {
  return new Promise((resolve, reject) => {
    const access_token = common_vendor.index.getStorageSync("access_token");
    if (!access_token) {
      common_vendor.index.__f__("log", "at utils/request.js:174", "[UPLOAD] ❌ 未登录");
      common_vendor.index.showToast({
        title: "请先登录",
        icon: "none"
      });
      reject({
        code: 401,
        message: "未登录"
      });
      return;
    }
    const {
      path = "tmp",
      name = "file",
      on_progress = null
    } = options;
    common_vendor.index.__f__("log", "at utils/request.js:193", `[UPLOAD] 📤 开始上传文件: ${temp_file_path}`), common_vendor.index.__f__("log", "at utils/request.js:194", `请求地址: ${base_url}/upload_image`);
    common_vendor.index.__f__("log", "at utils/request.js:195", `token ${access_token}`);
    common_vendor.index.__f__("log", "at utils/request.js:196", `formData: {'upload_path':${path}};`);
    const upload_task = common_vendor.index.uploadFile({
      url: `${base_url}/upload_image`,
      filePath: temp_file_path,
      name: "file",
      header: {
        "Authorization": `Bearer ${access_token}`
      },
      formData: {
        upload_path: path
      },
      success: async (res) => {
        common_vendor.index.__f__("log", "at utils/request.js:209", "✅ 上传成功 - 原始响应:", res);
        let response_data;
        try {
          response_data = JSON.parse(res.data);
          common_vendor.index.__f__("log", "at utils/request.js:213", "🧠 解析后的响应:", response_data);
        } catch (e) {
          common_vendor.index.__f__("error", "at utils/request.js:215", "❌ 解析失败:", e);
          common_vendor.index.showToast({
            title: "解析失败",
            icon: "none"
          });
          reject({
            success: false,
            message: "服务器返回格式错误"
          });
          return;
        }
        if (response_data.code === 200) {
          const image_url = `${static_base}/${response_data.url}`;
          resolve({
            success: true,
            data: {
              ...response_data,
              image_url,
              // 完整可访问 URL
              relative_url: response_data.url
              // 相对路径
            }
          });
        } else if (response_data.code === 401) {
          common_vendor.index.__f__("log", "at utils/request.js:240", `[HTTP] 🔑 Token过期，准备刷新...`);
          try {
            await refresh_token_def();
            common_vendor.index.__f__("log", "at utils/request.js:243", `[HTTP] 🔄 刷新成功，重试请求: ${api}`);
            return await upload_file(temp_file_path, options);
          } catch (err) {
            common_vendor.index.__f__("log", "at utils/request.js:246", "[HTTP] ❌ 刷新失败，跳转登录");
            common_vendor.index.removeStorageSync("access_token");
            common_vendor.index.removeStorageSync("refresh_token");
            common_vendor.index.removeStorageSync("token_expire");
            common_vendor.index.redirectTo({
              url: "/pages/profile/login/login?showToastTitle=需要重新登录"
            });
            throw {
              code: 401,
              message: "认证失败，已跳转登录"
            };
          }
        } else {
          common_vendor.index.showToast({
            title: response_data.message || "上传失败",
            icon: "none",
            duration: 2e3
          });
          reject({
            success: false,
            message: response_data.message
          });
        }
      },
      fail: (err) => {
        common_vendor.index.__f__("error", "at utils/request.js:272", "上传失败:", err);
        common_vendor.index.showToast({
          title: "网络错误或服务器无响应",
          icon: "none"
        });
        reject({
          success: false,
          message: "网络错误"
        });
      }
    });
    if (typeof on_progress === "function") {
      upload_task.onProgressUpdate((progress_res) => {
        on_progress(
          progress_res.progress,
          // 当前进度 %
          progress_res.totalBytesSent,
          // 已发送字节数
          progress_res.totalBytesExpectedToSend
          // 总字节数
        );
      });
    }
  });
}
exports.http_request = http_request;
exports.refresh_token_def = refresh_token_def;
exports.upload_file = upload_file;
//# sourceMappingURL=../../.sourcemap/mp-weixin/utils/request.js.map
